/*
 * VideoController.java 15 fev 2010
 *
 * Copyright (c) 2010 Emmanuel PUYBARET / eTeks <info@eteks.com>. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.eteks.sweethome3d.viewcontroller;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.List;

import com.eteks.sweethome3d.model.AspectRatio;
import com.eteks.sweethome3d.model.Camera;
import com.eteks.sweethome3d.model.Home;
import com.eteks.sweethome3d.model.HomeEnvironment;
import com.eteks.sweethome3d.model.UserPreferences;
import com.eteks.sweethome3d.viewcontroller.PhotoController.Property;

/**
 * The controller of the video creation view.
 * @author Emmanuel Puybaret
 */
public class VideoController implements Controller {
  /**
   * The properties that may be edited by the view associated to this controller. 
   */
  // Modife fma
//  public enum Property {ASPECT_RATIO, FRAME_RATE, WIDTH, HEIGHT, QUALITY, CAMERA_PATH}
  public enum Property {ASPECT_RATIO, FRAME_RATE, WIDTH, HEIGHT, QUALITY, CAMERA_PATH, YEAR, MONTH, DAY, HOUR}
  // Fin modife fma

  private final Home                  home;
  private final UserPreferences       preferences;
  private final ViewFactory           viewFactory;
  private final ContentManager        contentManager;
  private final PropertyChangeSupport propertyChangeSupport;
  private DialogView                  videoView;
  
  private AspectRatio                 aspectRatio;
  private int                         frameRate;
  private int                         width;
  private int                         height;
  private int                         quality;
  private List<Camera>                cameraPath;

  // Modife fma
  private int                         year;
  private int                         month;
  private int                         day;
  private int                         hour;
  // Fin modife fma

  public VideoController(Home home,
                         UserPreferences preferences, 
                         ViewFactory viewFactory,
                         ContentManager contentManager) {
    this.home = home;
    this.preferences = preferences;
    this.viewFactory = viewFactory;
    this.contentManager = contentManager;
    this.propertyChangeSupport = new PropertyChangeSupport(this);
    
    updateProperties();
  }

  /**
   * Returns the view associated with this controller.
   */
  public DialogView getView() {
    // Create view lazily only once it's needed
    if (this.videoView == null) {
      this.videoView = this.viewFactory.createVideoView(this.home, this.preferences, this);
    }
    return this.videoView;
  }

  /**
   * Displays the view controlled by this controller.
   */
  public void displayView(View parentView) {
    getView().displayView(parentView);
  }

  /**
   * Returns the content manager of this controller.
   */
  public ContentManager getContentManager() {
    return this.contentManager;
  }

  /**
   * Adds the property change <code>listener</code> in parameter to this controller.
   */
  public void addPropertyChangeListener(Property property, PropertyChangeListener listener) {
    this.propertyChangeSupport.addPropertyChangeListener(property.name(), listener);
  }

  /**
   * Removes the property change <code>listener</code> in parameter from this controller.
   */
  public void removePropertyChangeListener(Property property, PropertyChangeListener listener) {
    this.propertyChangeSupport.removePropertyChangeListener(property.name(), listener);
  }

  /**
   * Updates edited properties from the video creation preferences.
   */
  protected void updateProperties() {
    HomeEnvironment homeEnvironment = this.home.getEnvironment();
    setAspectRatio(homeEnvironment.getVideoAspectRatio());
    setFrameRate(homeEnvironment.getVideoFrameRate());
    setWidth(homeEnvironment.getVideoWidth(), false);
    setHeight(homeEnvironment.getVideoHeight(), false);
    setQuality(homeEnvironment.getVideoQuality());
    setCameraPath(homeEnvironment.getVideoCameraPath());

    // Modife fma
    setYear(homeEnvironment.getYear());
    setMonth(homeEnvironment.getMonth());
    setDay(homeEnvironment.getDay());
    setHour(homeEnvironment.getHour());
    // Fin modife fma
  }
  
  /**
   * Sets the aspect ratio of the video.
   */
  public void setAspectRatio(AspectRatio aspectRatio) {
    if (this.aspectRatio != aspectRatio) {
      AspectRatio oldAspectRatio = this.aspectRatio;
      this.aspectRatio = aspectRatio;
      this.propertyChangeSupport.firePropertyChange(Property.ASPECT_RATIO.name(), oldAspectRatio, aspectRatio);
      this.home.getEnvironment().setVideoAspectRatio(this.aspectRatio);
      setHeight(Math.round(width / this.aspectRatio.getValue()), false);
    }
  }
  
  /**
   * Returns the aspect ratio of the video.
   */
  public AspectRatio getAspectRatio() {
    return this.aspectRatio;
  }

  /**
   * Sets the frame rate of the video.
   */
  public void setFrameRate(int frameRate) {
    if (this.frameRate != frameRate) {
      int oldFrameRate = this.frameRate;
      this.frameRate = frameRate;
      this.propertyChangeSupport.firePropertyChange(Property.QUALITY.name(), oldFrameRate, frameRate);
      this.home.getEnvironment().setVideoFrameRate(this.frameRate);
    }
  }
  
  /**
   * Returns the frame rate of the video.
   */
  public int getFrameRate() {
    return this.frameRate;
  }

  /**
   * Sets the width of the video.
   */
  public void setWidth(int width) {
    setWidth(width, true);
  }
  
  private void setWidth(int width, boolean updateHeight) {
    if (this.width != width) {
      int oldWidth = this.width;
      this.width = width;
      this.propertyChangeSupport.firePropertyChange(Property.WIDTH.name(), oldWidth, width);
      if (updateHeight) {
        setHeight(Math.round(width / this.aspectRatio.getValue()), false);
      }
      this.home.getEnvironment().setVideoWidth(this.width);
    }
  }
  
  /**
   * Returns the width of the video.
   */
  public int getWidth() {
    return this.width;
  }

  /**
   * Sets the height of the video.
   */
  public void setHeight(int height) {
    setHeight(height, true);
  }
  
  private void setHeight(int height, boolean updateWidth) {
    if (this.height != height) {
      int oldHeight = this.height;
      this.height = height;
      this.propertyChangeSupport.firePropertyChange(Property.HEIGHT.name(), oldHeight, height);
      if (updateWidth) {
        setWidth(Math.round(height * this.aspectRatio.getValue()), false);
      }
    }
  }
  
  /**
   * Returns the height of the video.
   */
  public int getHeight() {
    return this.height;
  }

  /**
   * Sets the rendering quality of the video.
   */
  public void setQuality(int quality) {
    if (this.quality != quality) {
      int oldQuality = this.quality;
      this.quality = Math.min(quality, getQualityLevelCount() - 1);
      this.propertyChangeSupport.firePropertyChange(Property.QUALITY.name(), oldQuality, quality);
      this.home.getEnvironment().setVideoQuality(this.quality);
    }
  }
  
  /**
   * Returns the rendering quality of the video.
   */
  public int getQuality() {
    return this.quality;
  }

  /**
   * Returns the maximum value for quality.
   */
  public int getQualityLevelCount() {
    return 4;
  }

  /**
   * Returns the camera path of the video.
   */
  public List<Camera> getCameraPath() {
    return this.cameraPath;
  }
  
  /**
   * Sets the camera locations of the video.
   */
  public void setCameraPath(List<Camera> cameraPath) {
    if (this.cameraPath != cameraPath) {
      List<Camera> oldCameraPath = this.cameraPath;
      this.cameraPath = cameraPath;
      this.propertyChangeSupport.firePropertyChange(Property.CAMERA_PATH.name(), oldCameraPath, cameraPath);
      this.home.getEnvironment().setVideoCameraPath(this.cameraPath);
    }
  }

  // Modife fma
  public void setYear(int year) {
    if (this.year != year) {
      int oldYear = this.year;
      this.year = year;
      this.propertyChangeSupport.firePropertyChange(Property.YEAR.name(), oldYear, year);
      this.home.getEnvironment().setYear(this.year);
    }
  }

  public int getYear() {
    return this.year;
  }

  public void setMonth(int month) {
    if (this.month != month) {
      int oldMonth = this.month;
      this.month = month;
      this.propertyChangeSupport.firePropertyChange(Property.MONTH.name(), oldMonth, month);
      this.home.getEnvironment().setMonth(this.month);
    }
  }

  public int getMonth() {
    return this.month;
  }

  public void setDay(int day) {
    if (this.day != day) {
      int oldDay = this.day;
      this.day = day;
      this.propertyChangeSupport.firePropertyChange(Property.DAY.name(), oldDay, day);
      this.home.getEnvironment().setDay(this.day);
    }
  }

  public int getDay() {
    return this.day;
  }

  public void setHour(int hour) {
    if (this.hour != hour) {
      int oldHour = this.hour;
      this.hour = hour;
      this.propertyChangeSupport.firePropertyChange(Property.HOUR.name(), oldHour, hour);
      this.home.getEnvironment().setHour(this.hour);
    }
  }

  public int getHour() {
    return this.hour;
  }
// Fin modife fma
}
