/*
 * Sun.java 18 mars 2010
 *
 * Copyright (c) 2010 Frdric Mantegazza. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.eteks.sweethome3d.model;

import java.lang.Math;


public class Sun {
  private final double latitude;
  private final double longitude;
  private final int timezone;

  public Sun(double latitude, double longitude, int timezone) {
    this.latitude = latitude;
    this.longitude = longitude;
    this.timezone = timezone;
}

  public double computeJulianDay(int year, int month, int day, int hour, int minute, int second, int savingTime) {
    double day_ = day + hour / 24. + minute / 1440. + second / 86400.;
    if (month == 1 | month == 2) {
        year -= 1;
        month += 12;
    }
    int a = (int)(year / 100.);
    int b = 2 - a + (int)(a / 4.);

    double julianDay = (int)(365.25 * (year + 4716.)) + (int)((30.6001 * (month + 1))) + day_ + b - 1524.5;
    julianDay -= (this.timezone + savingTime) / 24.;
    julianDay -= 2451545.;

    return julianDay;
  }

  public double siderealTime(double julianDay) {
    double centuries = julianDay / 36525.;
    double siderealTime = (24110.54841 + (8640184.812866 * centuries) + (0.093104 * Math.pow(centuries, 2.)) - (0.0000062 * Math.pow(centuries, 3.))) / 3600.;
    siderealTime = ((siderealTime / 24.) - (int)(siderealTime / 24.)) * 24.;

    return siderealTime;
  }

  public double[] equatorialCoordinates(int year, int month, int day, int hour, int minute, int second, int savingTime) {
    double julianDay = computeJulianDay(year, month, day, hour, minute, second, savingTime);

    double g = 357.529 + 0.98560028 * julianDay;
    double q = 280.459 + 0.98564736 * julianDay;
    double l = q + 1.915 * Math.sin(Math.toRadians(g)) + 0.020 * Math.sin(Math.toRadians(2 * g));
    double e = 23.439 - 0.00000036 * julianDay;
    double rightAscension = Math.toDegrees(Math.atan(Math.cos(Math.toRadians(e)) * Math.sin(Math.toRadians(l)) / Math.cos(Math.toRadians(l)))) / 15.;
    if (Math.cos(Math.toRadians(l)) < 0.) {
      rightAscension += 12.;
    }
    if ((Math.cos(Math.toRadians(l)) > 0.) & (Math.sin(Math.toRadians(l)) < 0.)) {
      rightAscension += 24.;
    }

    double declination = Math.toDegrees(Math.asin(Math.sin(Math.toRadians(e)) * Math.sin(Math.toRadians(l))));

    double result[];
    result = new double[2];
    result[0] = rightAscension;
    result[1] = declination;
    return result;
  }

  public double[] azimuthalCoordinates(int year, int month, int day, int hour, int minute, int second, int savingTime) {
    double julianDay =  computeJulianDay(year, month, day, hour, minute, second, savingTime);
    double siderealTime = siderealTime(julianDay);
    double angleH = 360. * siderealTime / 23.9344;
    double angleT = (hour - (this.timezone + savingTime) - 12. + minute / 60. + second / 3600.) * 360. / 23.9344;
    double angle = angleH + angleT;
    double[] dummy = equatorialCoordinates(year, month, day, hour, minute, second, savingTime);
    double rightAscension = dummy[0];
    double declination =   dummy[1];
    double angle_horaire = angle - rightAscension * 15. + this.longitude;

    double elevation = Math.toDegrees(Math.asin(Math.sin(Math.toRadians(declination)) * Math.sin(Math.toRadians(this.latitude)) - Math.cos(Math.toRadians(declination)) * Math.cos(Math.toRadians(this.latitude)) * Math.cos(Math.toRadians(angle_horaire))));

    double azimuth = Math.toDegrees(Math.acos((Math.sin(Math.toRadians(declination)) - Math.sin(Math.toRadians(this.latitude)) * Math.sin(Math.toRadians(elevation))) / (Math.cos(Math.toRadians(this.latitude)) * Math.cos(Math.toRadians(elevation)))));
    double sinazimuth = (Math.cos(Math.toRadians(declination)) * Math.sin(Math.toRadians(angle_horaire))) / Math.cos(Math.toRadians(elevation));
    if (sinazimuth < 0.) {
      azimuth = 360. - azimuth;
    }

    double[] result;
    result = new double[2];
    result[0] = elevation;
    result[1] = azimuth;
    return result;
  }
}
